#! /bin/bash -p

#
# Install/Uninstall script for ThinPrint client
# Execute the script with -h or --help flag to display information about how to execute this script.
# Optionally, see the print_usage() function below for the same details.
#

print_usage() {
   echo -e "thnucSetup_VMW_VHV_Mac.sh:\nThis script installs/uninstalls the ThinPrint filter cups and types on your MAC."
   echo -e "Please note that you will require root provileges to execute this script.\n"

   echo -e "Usage:\n"
   echo "Pass -i <thnuclnt_dir> or --install <thnuclnt_dir>: You must pass the absolute path of the thnuclnt directory from which you want to these install filter types"
   echo "Example: sudo ./thnuSetup_VMW_VHV.sh -i /Applications/VMware\ Horizon\ View\ Client.app/Contents/Library/thnuclnt"
   echo "Optionally, an additional option -f or --force can be appended as the third argument."
   echo "When -f is specified, we will overwrite already existing thinprint thnucups and thinprint filter types."
   echo "Note that invoking with -f or --force - the install action would be irreversible."
   echo -e "\nPass -u or --uninstall: Note that this action is irreversible!"
   echo -e "\nPass -h or --help to display this usage message."
   echo -e "\nPass -v or --version to display the version.\n"
}

VERSION=1.0
OVERWRITE=0

#
# View-Client only copies over the x86_64-darwin/ from the stock thinprint drop
# So set architecture to x86_64-darwin directly
#
TP_FILTER="thnucups"
TP_TYPES="thnuclnt.types"
TP_CONVS="thnuclnt.convs"

#
# Following globals are required to check if cups is installed or not
#
CUPS_CMD="cups-config"
CUPS_CMD_LOC="type -p $CUPS_CMD"
CUPS_SERVERBIN="--serverbin"
CUPS_SERVERROOT="--serverroot"
CUPS_DATADIR="--datadir"

#
# check_cups():
# This function checks if cups is installed on the system or not
# We execute "cups-config --version"
# ThinPrint can not work if "--serverbin", "--serverroot" and "--datadir" are not present
#
check_cups() {
if [ "$CUPS_CMD_LOC" = "" ]; then
   echo -e "cups is not installed - it is required by thnuclnt"
   echo -e "Exiting...."
   exit 1
fi
CUPS_VERSION=`$CUPS_CMD --version`
echo "CUPS Version installed: $CUPS_VERSION"

for iterDir in "$CUPS_SERVERBIN" "$CUPS_SERVERROOT" "$CUPS_DATADIR" ; do
   retDir=`$CUPS_CMD $iterDir`
   if [ ! -d "$retDir" ] ; then
      echo "cups $iterDir does not exist!"
      echo "Exiting..."
      exit 1
   fi
done;
}

#
# install_thnu_cups():
# This function installs the thinprint cups filters
# One required parameter: the THNUCLNT_DIR i.e. the directory from which to install thnucups
#
install_thnu_cups() {
#
# The default path for cups filters is /usr/lib/cups/filter/
# However, it looks like this may be different for different MAC OS X versions
# For 10.7.4 that I am testing on, it is at /usr/libexec/cups/filter/
# Safer to just read the output of cups-config --serverbin
#
CUPS_FILTER_PATH=`$CUPS_CMD $CUPS_SERVERBIN`
CUPS_TP_DEST="$CUPS_FILTER_PATH/filter/$TP_FILTER"
CUPS_TP_SRC=$1/"$TP_FILTER"

if [ -f "$CUPS_TP_DEST" ] && [ "$OVERWRITE" != "1" ] ; then
   echo "ThinPrint Cups filter (thnucups) already present - not overwriting."
elif [ -f "$CUPS_TP_DEST" ] && [ "$OVERWRITE" == "1" ] ; then
   echo "ThinPrint Cups filter (thnucups) already present - overwriting it."
   cp "$CUPS_TP_SRC" "$CUPS_TP_DEST"
else
   echo "ThinPrint Cups filter (thnucups) not present - installing it."
   cp "$CUPS_TP_SRC" "$CUPS_TP_DEST"
fi
}

#
# install_thnu_filter_types():
# This function installs the thinprint filter types and convs files
# One required parameter: the THNUCLNT_DIR i.e. the directory from which to install filter types
#
install_thnu_filter_types() {
CUPS_FILTER_TYPES_PATH=`$CUPS_CMD $CUPS_SERVERROOT`
CUPS_TP_TYPES_DEST="$CUPS_FILTER_TYPES_PATH/$TP_TYPES"
CUPS_TP_TYPES_SRC=$1/"$TP_TYPES"

if [ -f "$CUPS_TP_TYPES_DEST" ] && [ "$OVERWRITE" != "1" ] ; then
   echo "ThinPrint Cups filter types (thnuclnt.types) already present - not overwriting."
elif [ -f "$CUPS_TP_TYPES_DEST" ] && [ "$OVERWRITE" == "1" ] ; then
   echo "ThinPrint Cups filter types (thnuclnt.types) already present - overwriting it."
   cp "$CUPS_TP_TYPES_SRC" "$CUPS_TP_TYPES_DEST"
else
   echo "ThinPrint Cups filter types (thnuclnt.types) not present - installing it."
   cp "$CUPS_TP_TYPES_SRC" "$CUPS_TP_TYPES_DEST"
fi

CUPS_FILTER_CONVS_PATH=`$CUPS_CMD $CUPS_SERVERROOT`
CUPS_TP_CONVS_DEST="$CUPS_FILTER_CONVS_PATH/$TP_CONVS"
CUPS_TP_CONVS_SRC=$1/$TP_CONVS

if [ -f "$CUPS_TP_CONVS_DEST" ] && [ "$OVERWRITE" != "1" ] ; then
   echo "ThinPrint Cups filter types (thnuclnt.convs) already present - not overwriting."
elif [ -f "$CUPS_TP_CONVS_DEST" ] && [ "$OVERWRITE" == "1" ] ; then
   echo "ThinPrint Cups filter types (thnuclnt.convs) already present - overwriting it."
   cp "$CUPS_TP_CONVS_SRC" "$CUPS_TP_CONVS_DEST"
else
   echo "ThinPrint Cups filter types (thnuclnt.convs) not present - installing it."
   cp "$CUPS_TP_CONVS_SRC" "$CUPS_TP_CONVS_DEST"
fi
}

#
# do_uninstall():
# This function uninstalls the existing thnprint thnucups and filters
# This action is irreversible
# Note: This uninstalls the thinprint cups filters and filter types.
#       To make thinprint work again with either VMware Fusion VMware View, run this script again
#       with "--install <thnuclnt_dir>" option.
#
do_uninstall() {
CUPS_FILTER_PATH=`$CUPS_CMD $CUPS_SERVERBIN`
CUPS_TP_DEST="$CUPS_FILTER_PATH/filter/$TP_FILTER"

CUPS_FILTER_TYPES_PATH=`$CUPS_CMD $CUPS_SERVERROOT`
CUPS_TP_TYPES_DEST="$CUPS_FILTER_TYPES_PATH/$TP_TYPES"

CUPS_FILTER_CONVS_PATH=`$CUPS_CMD $CUPS_SERVERROOT`
CUPS_TP_CONVS_DEST="$CUPS_FILTER_CONVS_PATH/$TP_CONVS"

echo -e "Important! \nThis actions is uninstalling thinprint cups filters and filter types."
echo -e "To enable thinprint functionality with either VMware Fusion VMware View, run this script again \nwith \"--install <thnuclnt_dir>\" option.\n"

rm -f "$CUPS_TP_DEST" > /dev/null 2>&1
rm -f "$CUPS_TP_TYPES_DEST" > /dev/null 2>&1
rm -f "$CUPS_TP_CONVS_DEST" > /dev/null 2>&1
}

#
# restart_cupsd():
# Restart the cupsd daemon by sending the SIGHUP
#
restart_cupsd() {
   echo "Restarting cupsd daemon...."
   killall -HUP cupsd
}


#
# echo PID to calling process
#
echo $$


if [ "$#" == "0" ] ; then
   echo -e "No Command line options specified!\n"
   print_usage
   exit
fi


#
# Root privileges required to execute this script
#
if [ ! "$EUID" = "0" ]; then
  echo "Root permission required!"
  exit 1
fi

opt=$1
#
# If the first command line argument is install, the second argument must be the thnuclnt directory
# Optionally, an additional "-f" or "--force" can be appended as third argument.
# When "-f" is specified, we will overwrite already existing thinprint thnucups and thinprint filter types.
# This action is irreversible.

case $opt in
   -v|--version)
      echo "install_VMW_Mac.sh Version - $VERSION"
      if [ "$#" -gt 1 ] ; then
         echo "Command line options specified after -v or --version are ignored."
         echo "Please run with --help to see details about how to execute this script"
         exit
      fi
      exit
   ;;

   -h|--help)
      print_usage
      exit
   ;;

   -u|--uninstall)
      do_uninstall
      restart_cupsd
      if [ "$#" -gt 1 ] ; then
         echo "Command line options specified after -u or --uninstall are ignored."
         echo "Please run with --help to see details about how to execute this script"
         exit
      fi
      exit
   ;;

   -i|--install)
      if [ "$2" == "" ] ; then
         echo "You must provide absolute path to thnuclnt directory to install from!"
         exit
      fi
      THNUCLNT_DIR="$2"
      if [ ! -d "$THNUCLNT_DIR" ] ; then
         echo "$THNUCLNT_DIR does not exist!"
         echo "Please provide a correct path to the thnuclnt directory"
         exit
      fi
      if [ "$3" != "" ] && [ "$3" == "-f" ] || [ "$3" == "--force" ] ; then
         OVERWRITE=1
      fi
      check_cups
      install_thnu_cups "$THNUCLNT_DIR"
      install_thnu_filter_types "$THNUCLNT_DIR"
      restart_cupsd
   ;;

   -f|--force)
      echo "The -f or --force option can only be specified after -i <thnuclnt_dir>"
      echo "Example: sudo ./thnuSetup_VMW_VHV_Mac.sh -i <thnuclnt_dir> -f"
      exit
   ;;

   *)
      echo -e "Unrecognized command line option! \n"
      print_usage
      exit
   ;;
esac

