"""
Copyright 2008 VMware, Inc.  All rights reserved. -- VMware Confidential

Basic Core exceptions and exit codes.
"""

# Exit codes
EXIT_OK = 0
EXIT_ERROR = 1
EXIT_ROOT_REQUIRED = 2
EXIT_MULTIPLE_INSTALLERS = 3
EXIT_DOWNGRADE_ERROR = 4
EXIT_NO_COMPONENT_INSTALLED = 5
EXIT_CANCELLED = 6
EXIT_OUT_OF_SPACE = 7

# Generic Errors
class Error(Exception):
   """ Base core error """
   pass

class InstallError(Error):
   """ Generic installation error """
   pass

class ValidationError(Error):
   """ Raised when a validation fails """
   pass

class ValidationErrorNonFatal(Error):
   """ Raised when a validation fails, but it should only warn, not block installation. """
   pass

class EULAError(ValidationError):
   """ Raised when EULA is not accepted """
   pass

class EULADeclinedError(EULAError):
   """ Raised when EULA is explicitly declined """
   pass

class AbortError(Error, KeyboardInterrupt):
   """
   Raised when installation is aborted

   This exception is sub-classed from KeyboardInterrupt because some
   modules, like logging, explicitly check for exceptions like
   KeyboardInterrupt and SystemExit and let them percolate up the call
   stack and silence all others.
   """
   pass

class InstallerExit(Error):
   """ Raised by the GUI when it's ready to exit """
   pass


# Database Errors
class DatabaseError(StandardError):
   """ Base exception for generic database errors. """
   pass

class DoesNotExistError(DatabaseError):
   """ Exception for entries that do not exist in the database. """
   pass

class CoreVersionDoesNotExistError(DatabaseError):
   """ Exception for a new component that has no associated core version.
   There must be an Installer core already installed on the system before
   adding a component to the system that requires it.
   """
   pass

class IntegrityError(DatabaseError):
   """
   Exception for when entries cannot be added due to violating
   constraints (from a UNIQUE index, for example.)
   """
   pass

class DependencyError(DatabaseError):
   """
   Exception for entries that cannot be removed because they
   are referenced by another entry (a file belonging to a
   component, for example.)
   """
   pass

class MultipleInstallersError(DatabaseError):
   """
   Exception raised when another installation is already running.
   """
   pass


# Dependency Errors
class InvalidInstallError(Exception):
   """
   Exception when the possible install scenario is not valid.
   ex: Attempting to uninstall the component we are trying to install.
   """
   pass

class ConflictError(Exception):
   """
   Exception when something is depended and conflicted on at the same
   time
   """
   pass

class UnsatisfiedDependency(Exception):
   """
   Exception when a dependency cannot be satisfied
   """
   pass

class DowngradeError(Exception):
   """
   Exception when a product would be downgraded.
   """
   pass

class VersionError(Exception):
   """
   Exception when the supplied version string not correct
   """
   pass

class CycleError(Exception):
   """
   Exception when cyclic dependencies exist in the graph
   """
   pass


# Component Errors
class ComponentError(RuntimeError):
   """ Standard component error """
   def __init__(self, message, component=None):
      self.component = component
      super(ComponentError, self).__init__(message)

class ComponentManifestError(ComponentError):
   """ Error in the component manifest """
   def __init__(self, message, component=None):
      self.component = component
      super(ComponentManifestError, self).__init__(message)

# Transaction Errors
class DirectoryExistsError(Exception):
   """ Extraction directory is already there """
   pass


# Installer Errors
class InstallerError(Exception):
   """ Generic installation error """
   pass

class MalformedMessage(Exception):
   """ Message not formed correctly """
   pass

class MethodNotFound(Exception):
   """ Requested method does not exist """
   pass

class IllegalParameters(Exception):
   """ Illegal Parameters were passed to the C functions """
   pass

class RemoteError(Exception):
   """ There was a problem accessing the remote component """
   pass


# Bundle Errors
class BundleError(RuntimeError):
   """ Exception when loading a bundle """
   pass
