"""
Copyright 2008 VMware, Inc.  All rights reserved. -- VMware Confidential

Environent for components.

@todo: disable __import__ somehow (can't do it by just excluding from exec)
"""
import imp
import os

from vmis.db import db

import vmis.core.questions

from vmis.core.installer import Installer, InstallerType, IType

from vmis.util.path import path
from vmis.util.log import getLog
from vmis.util.shell import run, output, Escape

from vmis.core.component import ComponentError, InstalledComponent

from vmis.core.dependency import Version
from vmis import VERSION, VMISPYVERSION

import vmis.core.files as files
import vmis.util.path as path
from vmis.core.installer import Installer, InstallerType, IType
import vmis.util.shell as shell

log = getLog('vmis.core.env')

class ComponentDestination(files.Destination):
   def __init__(self, dest, installer, perm=420, fileType=files.File):
      # Initialize ourselves as a Destination.
      # BINDIR, LIBDIR, etc. is now stored in self.rawText
      super(ComponentDestination, self).__init__(dest, perm=perm,
                                                 fileType=fileType)

      # Store the installer
      self.installer = installer

    # Override _expand
   def _expand(self):
      # Ship this off to the installer side if we have an installer set
      # in order to get the expanded value back.
      if self.installer:
         val = self.installer.GetFileValue(str(self.rawText))
      else:
         # Try the default installer on this module
         if self.installer is not None:
            val = self.installer.GetFileValue(str(self.rawText))
         else:
            #val = str(self.rawText)
            val = str(files.Destination(self))

      return val

      # Stolen from Destination and modified in order to override the type
   def __div__(self, divisor):
      ret = ComponentDestination(path.path(self.rawText)/divisor, self.installer,
                                    perm=self.perm, fileType=self.fileType)
      ret.perm = self.perm
      ret.fileType = self.fileType
      return ret

   def __unicode__(self):
      return self._expand()

   def __str__(self):
      return self._expand()

   def __repr__(self):
      return repr(self._expand())

def SetComponentModuleENV(mod, fileObj):
   env = {}
   env['ENV'] = os.environ
   env['Installer'] = Installer  # So the module knows about the Installer class
   # Add in access to files

   # Import errors
   import vmis.core.errors as errors
   env.update(errors.__dict__)

   # Import Version
   from vmis.core.version import Version
   env['Version'] = Version

   # Import regular expression parsing
   import re
   env['re'] = re

   for i in ['SystemFile', 'ConfigFile', 'File', 'Link', 'HardLink', 'Destination']:
      env[i] = getattr(files, i)
   for i in ['path']:
      env[i] = getattr(path, i)

   for txt in ['BINARY', 'SETUID', 'DEFAULT']:
      var = getattr(vmis.core.files, txt, None)
      env[txt] = var

   env['BINARY'] = 493
   env['SETUID'] = 2541
   env['DEFAULT'] = 420
   env['PYTHON_VERSION'] = '27'
   env['RET_VAL'] = 0
   env['RET_STDOUT'] = 1
   env['RET_STDERR'] = 2

   env['PREFIX'] = ComponentDestination('${prefix}', None, perm=420, fileType=files.File)
   env['SYSCONFDIR'] = ComponentDestination('${sysconfdir}', None, perm=420, fileType=files.File)
   env['BINDIR'] = ComponentDestination('${bindir}', None, perm=493, fileType=files.File)
   env['SBINDIR'] = ComponentDestination('${sbindir}', None, perm=493, fileType=files.File)
   env['LIBDIR'] = ComponentDestination('${libdir}', None, perm=420, fileType=files.File)
   env['DATADIR'] = ComponentDestination('${datadir}', None, perm=420, fileType=files.File)
   env['DOCDIR'] = ComponentDestination('${datadir}/doc', None, perm=420, fileType=files.File)
   env['MANDIR'] = ComponentDestination('${mandir}', None, perm=420, fileType=files.File)
   env['INCLUDEDIR'] = ComponentDestination('${includedir}', None, perm=420, fileType=files.File)
   env['INITSCRIPTDIR'] = ComponentDestination('${initscriptdir}', None, perm=493, fileType=files.File)
   env['INITDIR'] = ComponentDestination('${initdir}', None, perm=420, fileType=files.File)
   env['CONFDIR'] = ComponentDestination('${confdir}', None, perm=420, fileType=files.File)

   log.error(u'Before update: %s', str(mod.__dict__.values()))
   mod.__dict__.update(env)
   log.error(u'After update: %s', str(mod.__dict__.values()))

   log.error(u'Before exec: %s', str(mod.__dict__.values()))
   exec fileObj in mod.__dict__
   log.error(u'After exec: %s', str(mod.__dict__.values()))
   return env


def LoadInstaller(component, loadPath):
   """
   Load an installer module

   @param component: component to load
   @param loadPath: path to load it from
   @returns (module, installer): returns a tuple containing the containing
   module and installer object.  A reference to the module must be retained
   otherwise you'll wonder why things strangely go out of scope and disappear.

   @raises ImportError
   @raises ComponentError
   """
   # Some components have . in the name (like python2.5) which
   # Python will interpret as being a module separator

   moduleName = component.name.replace('.', '')

   fileObj, pathName, description = imp.find_module(moduleName, [loadPath])

   mod = imp.new_module(moduleName)
   env = SetComponentModuleENV(mod, fileObj)

   # Find subclass of installer
   installers = [o for o in mod.__dict__.values() if
              type(o) is InstallerType and o is not Installer]

   if len(installers) != 1:
      raise ComponentError(u'Component did not register an installer', component)
   else:
      installer = installers[0]()

   installer.SetInstallerType(IType.LOCALOPS) # We want this side to use local calls
   installer.SetComponent(component)
   installer.loadPath = loadPath


   # Move some functionality out of the installer, into the module.
   mod.__dict__['log'] = installer.log
   mod.__dict__['gui'] = installer.gui
   mod.__dict__['installer'] = installer
   mod.__dict__['PREFIX'].installer = installer
   mod.__dict__['SYSCONFDIR'].installer = installer
   mod.__dict__['BINDIR'].installer = installer
   mod.__dict__['SBINDIR'].installer = installer
   mod.__dict__['LIBDIR'].installer = installer
   mod.__dict__['DATADIR'].installer = installer
   mod.__dict__['DOCDIR'].installer = installer
   mod.__dict__['MANDIR'].installer = installer
   mod.__dict__['INCLUDEDIR'].installer = installer
   mod.__dict__['INITSCRIPTDIR'].installer = installer
   mod.__dict__['INITDIR'].installer = installer
   mod.__dict__['CONFDIR'].installer = installer

   # mod MUST be held onto so that Python does not garbage
   # collect this module.  Furthermore, this reference is
   # used in the component's LoadInclude method so that an
   # import can have a copy of the component's full environment.
   # Hold a reference to this component's environment as well.
   installer.mod = mod
   installer.env = env

   # XXX:  There is no module on this side, so we return None.  It is only being
   #  used to store a reference so that Python does not garbage collect it, and
   #  in this case, we have no module on this side.  It has been created and stored
   #  in the remote component.
   return (None, installer)
