#!/bin/bash

action="$1" # add, remove, or status
serviceName="$2"
startLevel="$3"
stopLevel="$4"

initConfigProgram() {
   # returns 'update-rc.d', 'insserv', 'chkconfig,
   #         or '' if not found.
   if [ -x /usr/sbin/update-rc.d ]; then
      echo '/usr/sbin/update-rc.d'
      return
   else
      if which update-rc.d >/dev/null 2>&1; then
         echo `which update-rc.d`
         return
      fi
   fi

   if [ -x /sbin/insserv ]; then
      echo '/sbin/insserv'
      return
   else
      if which insserv >/dev/null 2>&1; then
         echo `which insserv`
         return
      fi
   fi

   if [ -x /sbin/chkconfig ]; then
      echo '/sbin/chkconfig'
      return
   else
      if which chkconfig >/dev/null 2>&1; then
         echo `which chkconfig`
         return
      fi
   fi

   # Nothing found, return an empty string
   echo ''
}

findRCBaseDir() {
      rcDirs=""
      if [ -d /etc/rc0.d ]; then
         rcDirs="/etc"
      else
         if [ -d /etc/init.d/rc0.d ]; then
            rcDirs="/etc/init.d"
         else
            if [ -d /etc/rc.d/rc0.d ]; then
               rcDirs="/etc/rc.d"
            fi
         fi
      fi

      echo "$rcDirs"
}

findInitDir() {
      if [ -d /etc/init.d ]; then
         echo "/etc/init.d"
         return
      fi

      echo ""
}

addService() {
   initProg=`initConfigProgram`

   if [ "$initProg" = "" ]; then
      # Didn't find any of the handy 3 programs for adding init scripts.  Now try to do it manually.
      rcDirs=`findRCBaseDir`
      initDir=`findInitDir`

      # We've given it a good shot and haven't found the directories.  Give up.
      if [ -z "$rcDirs" -o -z "$initDir" ]; then
         echo "Cannot find insserv, chkconfig, or update-rc.d on your system.  Also cannot find System V style init directories.  Unable to add service."
         exit 1
      fi

      # Make sure startLevel and stopLevel fit the right format
      if [ "$startLevel" -lt 10 ]; then
         if [ "${startLevel:0:1}" != "0" ]; then
            startLevel="0${startLevel}"
         fi
      fi
      if [ "$stopLevel" -lt 10 ]; then
         if [ "${stopLevel:0:1}" != "0" ]; then
            stopLevel="0${stopLevel}"
         fi
      fi

      # Now set manual links into the correct directories for runlevels 2, 3, and 5
      for level in 2 3 5; do
         ln -s "$initDir"/$serviceName "$rcDirs/rc${level}.d/S${startLevel}${serviceName}"
         ln -s "$initDir"/$serviceName "$rcDirs/rc${level}.d/K${stopLevel}${serviceName}"
      done
   fi

   if [[ "$initProg" =~ "insserv" ]]; then
      if [ "$VMWARE_DEBUG" = "yes" ]; then
         "$initProg" -f $serviceName # Debug, allow the warning spew from Ubuntu
      else
         "$initProg" -f $serviceName 2>/dev/null  # Ubuntu will spew unrelated warnings..  Hide these.
      fi
      return
   fi

   if [[ "$initProg" =~ "chkconfig" ]]; then
      "$initProg" --add $serviceName
      return
   fi

   if [[ "$initProg" =~ "update-rc.d" ]]; then
      "$initProg" $serviceName start $startLevel 2 3 4 . stop $stopLevel 0 6 .
      return
   fi
}

removeService() {
   initProg=`initConfigProgram`

   if [ "$initProg" = "" ]; then
      # Didn't find any of the handy 3 programs for adding init scripts.  Now try to do it manually.
      rcDirs=`findRCBaseDir`
      initDir=`findInitDir`

      # We've given it a good shot and haven't found the directories.  Give up.
      if [ -z "$rcDirs" -o -z "$initDir" ]; then
         echo "Cannot find insserv, chkconfig, or update-rc.d on your system.  Also cannot find System V style init directories.  Unable to add service."
         exit 1
      fi

      # Now remove links from the directories for runlevels 2, 3, and 5
      for level in 2 3 5; do
         rm -f $rcDirs/rc${level}.d/S??${serviceName}
         rm -f $rcDirs/rc${level}.d/K??${serviceName}
      done
   fi

   if [[ "$initProg" =~ "insserv" ]]; then
      if [ "$VMWARE_DEBUG" = "yes" ]; then
         "$initProg" -f -r $serviceName # Debug, allow the warning spew from Ubuntu
      else
         "$initProg" -f -r $serviceName 2>/dev/null  # Ubuntu will spew unrelated warnings..  Hide these.
      fi
      return
   fi

   if [[ "$initProg" =~ "chkconfig" ]]; then
      "$initProg" --del $serviceName
      return
   fi

   if [[ "$initProg" =~ "update-rc.d" ]]; then
      "$initProg" -f $serviceName remove
      return
   fi
}

# Check to see whether a program is set to start on boot.
checkService() {
   # XXX: sysvinit checking only at the moment.  Add support for Upstart.

   # Search for start links in /etc.  find returns 0 no matter if it found
   # files or not, so run the output through grep to check if it really found
   # anything (also to quiet the output of find)
   # Restrict the search to /etc/rc*.d unless nothing is found.  Some systems put their
   # init scripts in /etc/init.d, so check in there for matching symlinks if the first
   # check fails.  Also send stderr to /dev/null... We don't want errors spewing out if the
   # find fails.
   (find /etc/rc*.d -type l -name "S??${serviceName}" 2>/dev/null | grep -q "${serviceName}") || \
      (find /etc/init.d -type l -name "S??${serviceName}" 2>/dev/null | grep -q "${serviceName}")

   retval=$?

   if [ "$retval" = "0" ]; then
      echo 'on'
      exit 5
   else
      echo 'off'
      exit 6
   fi
}

usage() {
      echo "Syntax for this script is as follows:"
      echo ""
      echo " $0 <action> <serviceName> <startLevel> <stopLevel>"
      echo ""
      echo " action      - add or remove"
      echo " serviceName - The name of the service (If it's /etc/init.d/samba, the name is just samba)"
      echo " startLevel  - Number of the start level"
      echo " stopLevel   - Number of the stop level"
      echo ""
      echo ""
      echo " $0 status <serviceName>"
      echo " serviceName - The name of the service (If it's /etc/init.d/samba, the name is just samba)"
      echo ""
}



case $action in
   add)
      # Don't allow any empty arguments for add
      if [ "$1" = "" ] || [ "$2" = "" ] || [ "$3" = "" ] || [ "$4" = "" ]; then
         usage
         exit 1
      fi
      addService
      ;;
   remove)
      # Don't allow any empty arguments for remove
      if [ "$1" = "" ] || [ "$2" = "" ] || [ "$3" = "" ] || [ "$4" = "" ]; then
         usage
         exit 1
      fi
      removeService
      ;;
   status)
      # We only need the serviceName to check status
      if [ "$1" = "" ] || [ "$2" = "" ]; then
         usage
         exit 1
      fi
      checkService
      ;;
   *)
      usage
      exit 1
      ;;
esac
