!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubestatistics_histo2d
  use cube_types
  use cubetools_parameters
  use cubetools_structure
  use cubetools_switch_types
  use cubeadm_cubeid_types
  use cubestatistics_messaging
  use cubestatistics_histoaxis_types
  !
  public :: histo2d
  public :: cubestatistics_histo2d_command
  private
  !
  type histo2d_t
     type(cube_t), pointer :: cube
     type(histoaxis_prog_t) :: x
     type(histoaxis_prog_t) :: y
     type(switch_prog_t) :: blank     ! Blank empty bins?
     type(switch_prog_t) :: normalize ! Normalize histogram?
  end type histo2d_t
  !
  type :: histo2d_comm_t
     type(option_t), pointer :: comm
     type(histoaxis_opt_t) :: x
     type(histoaxis_opt_t) :: y
     type(switch_opt_t) :: blank     
     type(switch_opt_t) :: normalize
   contains
     procedure, public  :: register => cubestatistics_histo2d_register
     procedure, private :: parse    => cubestatistics_histo2d_parse
     procedure, private :: main     => cubestatistics_histo2d_main
  end type histo2d_comm_t
  type(histo2d_comm_t) :: histo2d
  !
  integer(kind=4), parameter :: ixcube = 1 
  integer(kind=4), parameter :: iycube = 2
  type histo2d_user_t
     type(cubeid_user_t) :: cubeids
     type(histoaxis_user_t) :: x
     type(histoaxis_user_t) :: y
     type(switch_user_t) :: blank
     type(switch_user_t) :: normalize
   contains
     procedure, private :: toprog => cubestatistics_histo2d_user_toprog
  end type histo2d_user_t
  !
  type histo2d_prog_t
     type(cube_t), pointer :: xcube   ! Input  cube used as histo2d x-axis
     type(cube_t), pointer :: ycube   ! Input  cube used as histo2d y-axis
     type(cube_t), pointer :: pointer ! Output cube storing the position of the pixel in the 2D histogram
     type(histo2d_t) :: histo2d       ! Output 2D histogram
   contains
     procedure, private :: header => cubestatistics_histo2d_prog_header
     procedure, private :: data   => cubestatistics_histo2d_prog_data
     procedure, private :: loop   => cubestatistics_histo2d_prog_loop
     procedure, private :: act    => cubestatistics_histo2d_prog_act
  end type histo2d_prog_t
  !
contains
  !
  subroutine cubestatistics_histo2d_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(histo2d_user_t) :: user
    character(len=*), parameter :: rname='HISTO2D>COMMAND'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    call histo2d%parse(line,user,error)
    if (error) return
    call histo2d%main(user,error)
    if (error) return
  end subroutine cubestatistics_histo2d_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubestatistics_histo2d_register(histo2d,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_comm_t), intent(inout) :: histo2d
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    character(len=*), parameter :: comm_abst = &
         'Compute the joint histogram of two cubes'
    character(len=*), parameter :: comm_help = &
         'Compute the joint histogram of two cubes'
    character(len=*), parameter :: rname='HISTO2D>REGISTER'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    ! Command
    call cubetools_register_command(&
         'HISTO2D','cube1 cube2',&
         comm_abst,&
         comm_help,&
         cubestatistics_histo2d_command,&
         histo2d%comm,error)
    if (error) return
    call cubearg%register(&
         'XCUBE',&
         'Input cube #1 used as X axis',  &
         strg_id,&
         code_arg_mandatory,&
         [flag_cube],&
         error)
    if (error) return
    call cubearg%register(&
         'YCUBE',&
         'Input cube #2 used as Y axis',&
         strg_id,&
         code_arg_mandatory,&
         [flag_cube],&
         error)
    if (error) return
    !
    call histo2d%x%register('X',error)
    if (error) return
    ! 
    call histo2d%y%register('Y',error)
    if (error) return
    ! 
    call histo2d%blank%register(&
         'BLANK','setting empty bins to NaN',&
         'ON',error)
    if (error) return
    ! 
    call histo2d%normalize%register(&
         'NORMALIZE','histogram normalization from counts to %',&
         'OFF',error)
    if (error) return
  end subroutine cubestatistics_histo2d_register
  !
  subroutine cubestatistics_histo2d_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_comm_t), intent(inout) :: comm
    character(len=*),      intent(in)    :: line
    type(histo2d_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='HISTO2D>PARSE'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,comm%comm,user%cubeids,error)
    if (error) return
    call comm%x%parse(line,user%x,error)
    if (error) return
    call comm%y%parse(line,user%y,error)
    if (error) return
    call comm%blank%parse(line,user%blank,error)
    if (error) return
    call comm%normalize%parse(line,user%normalize,error)
    if (error) return
  end subroutine cubestatistics_histo2d_parse
  !
  subroutine cubestatistics_histo2d_main(comm,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_comm_t), intent(in)    :: comm
    type(histo2d_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(histo2d_prog_t) :: prog
    character(len=*), parameter :: rname='HISTO2D>MAIN'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubestatistics_histo2d_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubestatistics_histo2d_user_toprog(user,comm,prog,error)
    use cubetools_consistency_methods
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_user_t), intent(in)    :: user
    type(histo2d_comm_t),  intent(in)    :: comm
    type(histo2d_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    logical :: prob
    character(len=*), parameter :: rname='HISTO2D>USER2PROG'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(histo2d%comm,ixcube,user%cubeids,code_access_imaset,code_read,prog%xcube,error)
    if (error) return
    call cubeadm_cubeid_get_header(histo2d%comm,iycube,user%cubeids,code_access_imaset,code_read,prog%ycube,error)
    if (error) return
    !
    prob = .false.
    call cubetools_consistency_shape('Input cube #1',prog%xcube%head,'Input cube #2',prog%ycube%head,prob,error)
    if (error) return
    if (cubetools_consistency_failed(rname,prob,error)) return
    !
    call user%x%toprog(comm%x,prog%xcube,prog%histo2d%x,error)
    if (error) return
    call user%y%toprog(comm%y,prog%ycube,prog%histo2d%y,error)
    if (error) return
    call prog%histo2d%x%list(error)
    if (error) return
    call prog%histo2d%y%list(error)
    if (error) return
    !
    call prog%histo2d%normalize%init(comm%normalize,error)
    if (error) return
    call user%normalize%toprog(comm%normalize,prog%histo2d%normalize,error)
    if (error) return
    call prog%histo2d%blank%init(comm%blank,error)
    if (error) return
    call user%blank%toprog(comm%blank,prog%histo2d%blank,error)
    if (error) return
  end subroutine cubestatistics_histo2d_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubestatistics_histo2d_prog_header(prog,error)
    use cubetools_unit
    use cubetools_axis_types
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(axis_t) :: axis
    character(len=unit_l) :: unit
    character(len=*), parameter :: rname='HISTO2D>PROG>HEADER'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%xcube,[flag_histo2d,flag_pointer],prog%pointer,error)
    if (error) return
    call cubeadm_clone_header(prog%xcube,flag_histo2d,prog%histo2d%cube,error)
    if (error) return
    ! Unit
    if (prog%histo2d%normalize%enabled) then
       call cubetools_header_put_array_unit('%',prog%histo2d%cube%head,error)
       if (error) return
    else
       call cubetools_header_put_array_unit('Counts',prog%histo2d%cube%head,error)
       if (error) return
    endif
    ! X axis
    call cubetools_header_get_array_unit(prog%xcube%head,unit,error)
    if (error) return
    call cubetools_header_get_axis_head_l(prog%xcube%head,axis,error)
    if (error) return
    axis%name = 'X'
    if (prog%histo2d%x%dolog) then
       ! *** JP huge risk of overflow here...
       ! Try to decrease potential overflow by only adding 'log ' at start.
       axis%unit = 'log '//trim(unit)
    else
       axis%unit = unit
    endif
    axis%kind = unit_unk%id
    axis%genuine = .true.
    axis%regular = .true.
    axis%n   = prog%histo2d%x%n
    axis%ref = 1.0
    axis%val = prog%histo2d%x%min
    axis%inc = prog%histo2d%x%inc
    call cubetools_header_update_axset_l(axis,prog%histo2d%cube%head,error)
    if (error) return
    ! Y axis
    call cubetools_header_get_array_unit(prog%ycube%head,unit,error)
    if (error) return
    call cubetools_header_get_axis_head_m(prog%ycube%head,axis,error)
    if (error) return
    axis%name = 'Y'
    if (prog%histo2d%y%dolog) then
       ! *** JP huge risk of overflow here...
       ! Try to decrease potential overflow by only adding 'log ' at start.
       axis%unit = 'log '//trim(unit)
    else
       axis%unit = unit
    endif
    axis%kind = unit_unk%id
    axis%genuine = .true.
    axis%regular = .true.
    axis%n   = prog%histo2d%y%n
    axis%ref = 1.0
    axis%val = prog%histo2d%y%min
    axis%inc = prog%histo2d%y%inc
    call cubetools_header_update_axset_m(axis,prog%histo2d%cube%head,error)
    if (error) return
  end subroutine cubestatistics_histo2d_prog_header
  !
  subroutine cubestatistics_histo2d_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='HISTO2D>PROG>DATA'
    !
    call cubestatistics_message(statisticsseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       !$OMP TASK SHARED(prog,error) FIRSTPRIVATE(iter)
       if (.not.error) &
         call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubestatistics_histo2d_prog_data
  !
  subroutine cubestatistics_histo2d_prog_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubeadm_image_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: xima,yima,pima,histo2d
    character(len=*), parameter :: rname='HISTO2D>PROG>LOOP'
    !
    ! The allocation of xima and yima allows me to take the logarithm of the inputs
    call xima%allocate('xima',prog%xcube,error)
    if (error) return
    call yima%allocate('yima',prog%ycube,error)
    if (error) return
    call pima%allocate('pointer image',prog%pointer,error)
    if (error) return
    call histo2d%allocate('histo2d',prog%histo2d%cube,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act(ie,xima,yima,pima,histo2d,error)
      if (error) return
    enddo ! ie
  end subroutine cubestatistics_histo2d_prog_loop
  !
  subroutine cubestatistics_histo2d_prog_act(prog,ie,xima,yima,pima,histo2d,error)
    use cubetools_nan
    use cubeadm_image_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(histo2d_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t), target, intent(inout) :: xima
    type(image_t), target, intent(inout) :: yima
    type(image_t),         intent(inout) :: pima
    type(image_t),         intent(inout) :: histo2d
    logical,               intent(inout) :: error
    !
    integer(kind=pixe_k) :: ix,iy
    integer(kind=pixe_k) :: jx,jy
    integer(kind=data_k) :: nin,nou,nblank
    real(kind=sign_k), pointer :: xval,yval
    character(len=*), parameter :: rname='HISTO2D>ACT'
    !
    ! Get data
    call xima%get(ie,error)
    if (error) return
    call yima%get(ie,error)
    if (error) return
    ! Transform it
    if (prog%histo2d%x%dolog) then
       do iy=1,xima%ny
          do ix=1,xima%nx
             xval => xima%val(ix,iy)
             xval = log10(xval)
          enddo ! iy
       enddo ! ix
    endif
    if (prog%histo2d%y%dolog) then
       do iy=1,xima%ny
          do ix=1,xima%nx
             yval => yima%val(ix,iy)
             yval = log10(yval)
          enddo ! iy
       enddo ! ix
    endif
    ! Compute histogram
    nin = 0
    nou = 0
    nblank = 0
    histo2d%val = 0.0
    pima%val = 0
    do iy=1,xima%ny
       do ix=1,xima%nx
          xval => xima%val(ix,iy)
          yval => yima%val(ix,iy)
          if ((ieee_is_finite(xval)).and.(ieee_is_finite(yval))) then
             jx = nint((xval-prog%histo2d%x%min)/prog%histo2d%x%inc)
             jy = nint((yval-prog%histo2d%y%min)/prog%histo2d%y%inc)
             if ((1.le.jx).and.(jx.le.prog%histo2d%x%n).and.&
                  (1.le.jy).and.(jy.le.prog%histo2d%y%n)) then
                ! *** JP problem when many points?
                pima%val(ix,iy) = jx+(jy-1)*pima%nx
                histo2d%val(jx,jy) = histo2d%val(jx,jy)+1.0
                nin = nin+1
             else
                nou = nou+1
             endif
          else
             nblank = nblank+1
          endif
       enddo ! iy
    enddo ! ix
    ! Blank empty bins when asked
    if (prog%histo2d%blank%enabled) then
       do iy=1,prog%histo2d%y%n
          do ix=1,prog%histo2d%x%n
             if (histo2d%val(ix,iy).le.0.0) then
                histo2d%val(ix,iy) = gr4nan
             endif
          enddo ! iy
       enddo ! ix
    endif
    ! Normalize when asked
    if ((prog%histo2d%normalize%enabled).and.(nin.gt.0)) then
       do iy=1,prog%histo2d%y%n
          do ix=1,prog%histo2d%x%n
             histo2d%val(ix,iy) = 100d0*histo2d%val(ix,iy)/real(nin,kind=coor_k)
          enddo ! ix
       enddo ! iy
    endif
    ! Put result
    call pima%put(ie,error)
    if (error) return
    call histo2d%put(ie,error)
    if (error) return
  end subroutine cubestatistics_histo2d_prog_act
end module cubestatistics_histo2d
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
